/* Copyright (c) 2010 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT. 
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */ 

/** @file
 * @brief Implementation of lib_ir_receive
 */

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include "nrf24le1.h"
#include "lib_ir_receive.h"
#include "nordic_common.h"

// Internal variables
static uint16_t timer_counter_1;
static uint16_t timer_counter_2;
static uint16_t sequence_counter;
static uint8_t bit_counter;
static uint8_t pulse_counter;

static uint16_t gap_target;
static uint16_t timer_target;
static uint8_t timer_period;

static uint32_t data_sequence;

static uint8_t receive_flags;
static bool receive;
static bool first_edge_detected;

static int16_t tmp_pulse[2];
static int8_t current_input_level;

static lib_ir_protocol_struct_t current_protocol;

// Internal functions

/** Function to reset the counters, flags and protocol
 */    
static void reset(void);

/** Function to set the current protocol
 *  @param protocol The name of the protocol.
 */                 
static void set_protocol(lib_ir_protocol_name_t protocol);

/** Function to initiate a reception
 *  This function clears some variables and sets the receive flags
 */           
void initiate_reception(void);
           
void lib_ir_receive_init(uint8_t tp)
{ 
  timer_period = tp;
  
  reset();
}

static void reset(void)
{
  // Clear counters
  receive = false;
  first_edge_detected = false;
  timer_counter_1 = 0;
  timer_counter_2 = 0;
  timer_target = 0;
  data_sequence = 0;
  bit_counter = 0;
  pulse_counter = 0;

  // Clear the protocol
  set_protocol(PROTOCOL_NONE);
}

static void set_protocol(lib_ir_protocol_name_t protocol)
{ 
  // Set the correct protocol 
  switch (protocol)
  {
    case PROTOCOL_NONE:
    {
      memset(&current_protocol, 0x00, sizeof(lib_ir_protocol_struct_t));
      current_protocol.protocol = PROTOCOL_NONE;
      break;
    } 
    case PROTOCOL_NEC:
    {
      current_protocol = lib_ir_protocol_nec;
      break;
    }     
    case PROTOCOL_RC5:
    {
      current_protocol = lib_ir_protocol_rc5;
      break;
    }
    case PROTOCOL_SIRC_12BIT:
    {
      current_protocol = lib_ir_protocol_sirc_12bit;
      break;
    }
    case PROTOCOL_SIRC_15BIT:
    {
      current_protocol = lib_ir_protocol_sirc_15bit;
      break;
    } 
    case PROTOCOL_SIRC_20BIT:
    {
      current_protocol = lib_ir_protocol_sirc_20bit;
      break;
    } 
    default:
    {
      memset(&current_protocol, 0x00, sizeof(lib_ir_protocol_struct_t));
      break;
    }
  }

  // Calculate the timer target for the gap
  gap_target = (uint16_t)(current_protocol.gap / timer_period);
  sequence_counter = 0;
}

bool lib_ir_receive_command(lib_ir_protocol_name_t protocol)
{
  // Do not start any new reception while one is going on
  if (receive)
    return false;

  // Clear the data sequence so new data can be received
  data_sequence = 0;

  // Set the correct protocol if needed
  if (current_protocol.protocol != protocol)
    set_protocol(protocol);

  // State which part of the packet we are expecting
  receive_flags = current_protocol.flags;

  // Start the reception
  initiate_reception();

  return true;
}

void initiate_reception(void)
{ 
  // Set the number of bits we want to receive
  bit_counter = current_protocol.bits - 1;

  first_edge_detected = false;
  receive = true;
  pulse_counter = 0;
}

void lib_ir_receive_timer_irq_function(void)
{ 
  // If not in receive mode: Return from the function
  if(!receive)
    return;

  // If the first edge is not detected: Reset counters an return
  if(!first_edge_detected)
  {
    timer_counter_1 = 0;
    timer_counter_2 = 0;
    sequence_counter = 0;
    return;
  }

  // Increment the counters
  timer_counter_1++;
  timer_counter_2++;
  sequence_counter++;

  if(sequence_counter == gap_target)
  {
    // The data_sequence was not recognised on time and we flag an error
    reset();
    lib_ir_receive_error_callback();
    return;
  }

  // Enter the sequence for receiving the header if the flag is set
  if (receive_flags & HEADER)
  { 
    // If this is the first pulse of the header or if this is the second pulse and the timer has reached the target
    if (pulse_counter == 0 || (timer_counter_1 == timer_target && pulse_counter == 1))
    {
      // Set the timer target in the centre of the last header pulse 
      timer_target = (uint16_t)((abs(current_protocol.header[0]) + abs(current_protocol.header[1] / 2)) / timer_period); 

      // Save the level of the pulse and increment the pulse counter
      tmp_pulse[pulse_counter] = abs(current_protocol.header[pulse_counter]) * current_input_level;      
      pulse_counter++;
    }
    // When both header pulses are found
    if (pulse_counter == 2)
    {
      // Check if the pulse matches the pulse sequence for the header
      if (tmp_pulse[0] == current_protocol.header[0] && tmp_pulse[1] == current_protocol.header[1])
      {
        if (current_protocol.encoding == ENC_PDE) // Pulse Distance Encoding
        {
          // Set the timer target to the average of the '0' space and the '1' space
          // The input pin can be sampled at this point to determine the bit state
          timer_target = ((abs(current_protocol.one[1]) > abs(current_protocol.zero[1])) ? abs(current_protocol.one[1]) / 2 : abs(current_protocol.zero[1]) / 2) / timer_period;
        }
        else if (current_protocol.encoding == ENC_PLE) // Pulse Length Encoding
        {
          // Set the timer target to the average of the '0' pulse and the '1' pulse
          // The input pin can be sampled at this point to determine the bit state
          timer_target = (uint16_t)(((abs(current_protocol.one[0]) + abs(current_protocol.zero[0])) /2 ) / timer_period);
        }
        // Clear the header flag
        receive_flags &= ~HEADER;
      }
      // If the incoming pulse does not match, return an error            
      else
      {
        reset();
        lib_ir_receive_error_callback();
        return;
      }
      pulse_counter = 0;
    } 
  }

  // Enter the sequence for receiving the lead if the flag is set
  else if (receive_flags & LEAD)
  {
    // Check if the timer has reached the target
    if (timer_counter_1 == timer_target)
    {
      // This should be midway in the first pulse
      // Set the target equal to lead pulse width so that the lead pulse will not be sampled twice
      // This will be changed by the edge interrupt if an edge is detected
      timer_target = (uint16_t)((abs(current_protocol.lead)) / timer_period); 
      timer_counter_1 = 0;

      tmp_pulse[0] = abs(current_protocol.lead) * current_input_level;

      // Check if the pulse matches the pulse sequence for plead
      if (tmp_pulse[0] == current_protocol.lead)
      {
        pulse_counter = 0;
        // Clear the lead flag
        receive_flags &= ~LEAD;
      }
      // If the incoming pulse does not match, return an error            
      else
      {
        reset();
        lib_ir_receive_error_callback();
        return;
      }            
    }
  }

  // Enter the sequence for receiving the data if the flag is set
  else if (receive_flags & DATA)
  {
    if (current_protocol.encoding == ENC_PDE) // Pulse Distance Encoding
    {
      // Check if the timer has received the target
      if (timer_counter_2 == timer_target)
      {       
        // If the line is still high there is a long break => '1'
        if (current_input_level == LIB_IR_LEVEL_HIGH)
          data_sequence |= (uint32_t)1 << bit_counter;
          
        // If all the bits are detected: Clear the data flag, else decrement bit counter   
        if (bit_counter == 0)
          receive_flags &= ~DATA;
        else;
          bit_counter--;
      }      
    }
    else if (current_protocol.encoding == ENC_PLE) // Pulse Length Encoding
    {
      // Check if the timer has received the target
      if (timer_counter_1 == timer_target)
      {
        // If the line is still low there is a long pulse => '1'
        if (current_input_level == LIB_IR_LEVEL_LOW)
          data_sequence |= (uint32_t)1 << bit_counter; 
        
        // If all the bits are detected: Clear the data flag, else decrement bit counter  
        if (bit_counter == 0)
          receive_flags &= ~DATA;
        else;
          bit_counter--;
      }
    }
    else if (current_protocol.encoding == ENC_ME)  // Manchester Encoding
    { 
      // Check if the timer has received the target      
      if (timer_counter_1 == timer_target)
      {
        // The timer target should be in the middle of one pulse
        // Set the target equal to one pulse width so that if it is a double pulse you will sample twice
        // This will be changed by the edge interrupt if an edge is detected
        timer_target = (uint16_t)((abs(current_protocol.one[0])) / timer_period); 

        // Reset the counter
        timer_counter_1 = 0;

        if (pulse_counter <= 1)
        {
          // Save the last pulse and increment the counter
          tmp_pulse[pulse_counter] = abs(current_protocol.one[0]) * current_input_level;
          pulse_counter++;
        }
        if (pulse_counter == 2)
        {
          // When two pulses, check which state the bit has

          // Check if the pulse matches the pulse sequence for logical '1'
          if (tmp_pulse[0] == current_protocol.one[0] && tmp_pulse[1] == current_protocol.one[1])
            data_sequence |= 0x01 << bit_counter;
          // Check if the pulse matches the pulse sequence for logical '0'
          else if (tmp_pulse[0] == current_protocol.zero[0] && tmp_pulse[1] == current_protocol.zero[1])
            data_sequence |= 0x00 << bit_counter;
          else
          {
            // The bit was not recognized and we flag an error
            reset();
            lib_ir_receive_error_callback();
            return;
          }
          pulse_counter = 0;

          // If all the bits are detected: Clear the data flag, else decrement bit counter
          if (bit_counter == 0)
            receive_flags &= ~DATA;
          else;
            bit_counter--;
        }
      }
    }
  }

  // Enter the sequence for receiving the trail if the flag is set
  else if (receive_flags & TRAIL)
  {
    // Set the target to be in the middle of the trail pulse
    timer_target = (uint16_t)((abs(current_protocol.trail) / 2) / timer_period);
    if (timer_counter_1 == timer_target)
    {
      timer_counter_1 = 0;
      tmp_pulse[0] = abs(current_protocol.trail) * current_input_level;

      // Check if the pulse matches the pulse sequence for trail
      if (tmp_pulse[0] == current_protocol.trail)
      {
        // Clear the trail flag
        receive_flags &= ~TRAIL;
        pulse_counter = 0;
      }  
      // If the incoming pulse does not match, return an error            
      else
      {
        reset();
        lib_ir_receive_error_callback();
        return;
      }           
    }
  }
  // When the complete sequence is received, give a call-back and clear the receive flag
  else
  {
    lib_ir_receive_data_received_callback(data_sequence);
    receive = false;
  }
}

void lib_ir_receive_edge_irq_function(int8_t level)
{
  // If not in receive mode: Return from the function
  if (!receive)
    return;  

  // Save the level of the input line
  current_input_level = level;
  first_edge_detected = true;

  if (current_protocol.encoding == ENC_PDE) // Pulse Distance Encoding
  {  
    // Reset timer on high level (space)
    if(level == LIB_IR_LEVEL_HIGH)
    {
      timer_counter_2 = 0;
    }
    // Reset timer on high level (IR pulses)
    else if(level == LIB_IR_LEVEL_LOW)
    {
      timer_counter_1 = 0;
    }
  }
  else if (current_protocol.encoding == ENC_PLE) // Pulse Length Encoding
  {
    // Reset timer on low level (IR pulses)
    if(level == LIB_IR_LEVEL_LOW)
    {
      timer_counter_1 = 0;
    }
  }
  
  else if (current_protocol.encoding == ENC_ME)  // Manchester Encoding
  {
    // Set the timer target to the middle of a bit pulse
    timer_target = (uint16_t)(((abs(current_protocol.one[0])) /2 ) / timer_period);
    timer_counter_1 = 0;
  }
}
